/**
***	sudokuSolve.c
***
***	Letta in input una matrice 9x9 di numeri interi da un file,
***	verifica con un procedimento esaustiva l'esistenza di una 
***	soluzione della partita.
***
***	#(@) 20051204 (liverani@mat.uniroma3.it)
***
**/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/*
 *	leggiFile(S, M)
 *
 *	Legge da file di testo ASCII una configurazione del Sudoku
 *	e la memorizza nella matrice di interi.
 *
 *	S: stringa contenente il nome del file da aprire; se la stringa
 *	   e' nulla viene richiesto il nome del file all'utente.
 *
 *	M: matrice di interi 9x9 in cui viene memorizzata la configurazione.
 *
 *	Restituisce 1 se la lettura del file e` andata a buon fine,
 *	0 altrimenti.
 *
 */

int leggiFile(char *s, int M[9][9]) {
	FILE *f;
	int i, j, rc;

	while (s[0] == '\0') {
		printf("Nome del file: ");
		scanf("%s", s);
	}
	if ((f = fopen(s, "rt"))) {
		for (i=0; i<9 && !feof(f); i++) {
			for (j=0; j<9 && !feof(f); j++) {
				fscanf(f, "%d", &M[i][j]);
			}
		}
		if (i==9 && j==9) {
			rc = 1;
		} else {
			rc = 0;
		}
		fclose(f);
	} else {
		fprintf(stderr, "ERRORE: il file %s non puo' essere aperto in lettura\n\n", s);
		rc = 0;
	}
	return(rc);
}

/*
 *	stampaConfigurazione(M)
 *
 *	Stampa la matrice M 9x9 con una configurazione della
 *	scacchiera di gioco.
 *
 *	M: matrice 9x9 di interi
 *
 */
 
void stampaConfigurazione(int M[9][9]) {
	int i, j, h, k;

	printf("\n+---------+---------+---------+\n");
	for (i=0; i<3; i++) {
		for (j=3*i; j<3*(i+1); j++) {
			printf("|");
			for (k=0; k<3; k++) {
				for (h=3*k; h<3*(k+1); h++) {
					printf("%2d ", M[j][h]);
				}
				printf("|");
			}
			printf("\n");
		}
		printf("+---------+---------+---------+\n");
	}
	return;
}

/*
 *	usage
 *
 *	Visualizza le istruzioni per l'uso del programma.
 *
 */
 
void usage(void) {
	fprintf(stderr, "\n");
	fprintf(stderr, "SUDOKU: risolutore automatico di Sudoku.\n");
	fprintf(stderr, "        sudoku [-h | -f nomefile [-d] ]\n");
	fprintf(stderr, "        #(@) 20051204 (liverani@mat.uniroma3.it)\n\n");
	return;
}

/*
 *	verifica(M, i, j, debug)
 *
 *	Verifica che l'elemento M[i][j] contenga un valore compatibile
 *	con quelli presenti nella colonna j, nella riga i e nel riquadro
 *	3x3 in cui si trova l'elemento M[i][j].
 *
 *	M: matrice 9x9 di numeri interi con la configurazione di gioco.
 *
 *	i: indice di riga dell'elemento da verificare.
 *
 *	j: indice di colonna dell'elemento da verificare.
 *
 *	debug: se debug=1 visualizza messaggi di debug per seguire il
 *	   procedimento di calcolo.
 *
 *	La funzione restituisce 1 se la il valore di M[i][j] e' 
 *	compatibile con il resto della matrice, 0 altrimenti.
 *
 */
 
int verifica(int m[9][9], int i, int j, int debug) {
	int h, k, rc;

	rc = 1;
	for (h=0; h<9 && rc == 1; h++) {
		if (h != j && m[i][h] == m[i][j]) {
			rc = 0;
			if (debug)
				printf("m[%d][%d]=%d ... no: doppione in riga.\n", i, j, m[i][j]);
		} else {
			if (h != i && m[h][j] == m[i][j]) {
				rc = 0;
				if (debug)
					printf("m[%d][%d]=%d ... no: doppione in colonna.\n", i, j, m[i][j]);
			}
		}
	}
	for (h=3*(i/3); h<3*(i/3+1) && rc == 1; h++) {
		for (k=3*(j/3); k<3*(j/3+1) && rc == 1; k++) {
			if ((h != i || k != j) && m[h][k] == m[i][j]) {
				rc = 0;
				if (debug)
					printf("m[%d][%d]=%d ... no: doppione nel quadrato!\n", i, j, m[i][j]);
			}
		}
	}
	if (debug) {
		if (rc==1) {
			printf("Il valore m[%d][%d]=%d e' compatibile.\n", i, j, m[i][j]);
		} else {
			printf("Il valore m[%d][%d]=%d non e' compatibile.\n", i, j, m[i][j]);
		}
	}
	return(rc);
}

/*
 *	sudokuSolve(M, debug)
 *
 *	Funzione ricorsiva per la ricerca della configurazione risolutiva
 *	che prova a collocare nelle posizioni vuote tutti valori
 *	k=1, 2, ..., 9.
 *
 *	M: matrice 9x9 di interi con la configurazione corrente della 
 *	   griglia di gioco.
 *
 *	debug: se debug=1  visualizza i messaggi di debug per seguire 
 *	   l'evoluzione del processo di ricerca della soluzione.
 *
 *	La funzione restituisce 1 se la funzione riesce ad individuare
 *	una configurazione finale, 0 altrimenti (se la configurazione
 *	iniziale non ammette alcuna soluzione).
 *
 */
 
int sudokuSolve(int m[9][9], int debug) {
	int i, j, k, rc, ii=10, jj=10, flag;

	if (debug) {
		printf("\nProviamo a risolvere questa configurazione:");
		stampaConfigurazione(m);
	}
	rc = 0;

	flag = 0;
	for (i=0; i<9 && flag == 0; i++) {
		for (j=0; j<9 && flag == 0; j++) {
			if (m[i][j] == 0) {
				flag = 1;
				ii = i;
				jj = j;
			}
		}
	}
	if (flag == 1) {
		i = ii;
		j = jj;
	}

	if (i<9 && j<9) {
		for (k=1; k<10 && rc == 0; k++) {
			m[i][j] = k;
			if (verifica(m, i, j, debug) == 1 && sudokuSolve(m, debug) == 1) {
				rc = 1;
			}
		}
		if (rc == 0) {
			m[i][j] = 0;
			if (debug) {
				printf("Tutti i valori provati per m[%d][%d] non sono compatibili: modifichiamo i precedenti.\n", i, j);
			}
		}
	} else {
		printf("SUDOKU risolto!\n");
		stampaConfigurazione(m);
		rc = 1;
	}
	return(rc);
}

/*
 *	main
 *
 *	Funzione principale.
 *
 */
 
int main(int argc, char *argv[]) {
	char s[100];
	int m[9][9], i, debug = 0;

	s[0] = '\0';

	if (argc>0) {
		for (i=1; i<argc; i++) {
			if (strcmp(argv[i], "-h")==0 || strcmp(argv[i], "-?")==0) {
				usage();
				return(1);
			}
			if (strcmp(argv[i], "-f") == 0 && argc>i+1) {
				strcpy(s, argv[i+1]);
			}
			if (strcmp(argv[i], "-d") == 0) {
				debug = 1;
			}
		}
	}
	if (leggiFile(s, m) == 1) {
		stampaConfigurazione(m);
		if (sudokuSolve(m, debug) == 0) {
			printf("Questa configurazione del Sudoku non ammette soluzioni.\n");
		}
	}
	return(0);
}
